"""neutrino_api"""


class APIErrorCode:
    """
    Neutrino API error codes
    """
    INVALID_PARAMETER = 1
    MAX_CALL_LIMIT = 2
    BAD_URL = 3
    ABUSE_DETECTED = 4
    NOT_RESPONDING = 5
    CONCURRENT = 6
    NOT_VERIFIED = 7
    TELEPHONY_LIMIT = 8
    INVALID_JSON = 9
    ACCESS_DENIED = 10
    MAX_PHONE_CALLS = 11
    BAD_AUDIO = 12
    HLR_LIMIT_REACHED = 13
    TELEPHONY_BLOCKED = 14
    TELEPHONY_RATE_EXCEEDED = 15
    FREE_LIMIT = 16
    RENDERING_FAILED = 17
    DEPRECATED_API = 18
    CREDIT_LIMIT_REACHED = 19
    NOT_MULTI_ENABLED = 21
    NO_BATCH_MODE = 22
    BATCH_LIMIT_EXCEEDED = 23
    BATCH_INVALID = 24
    DATASET_UNKNOWN = 26
    DATASET_GENERATION_FAILED = 27
    DATASET_INVALID_DATE = 28
    USER_DEFINED_DAILY_LIMIT = 31
    ACCESS_FORBIDDEN = 43
    REQUEST_TOO_LARGE = 44
    NO_ENDPOINT = 45
    INTERNAL_SERVER_ERROR = 51
    SERVER_OFFLINE = 52
    CONNECT_TIMEOUT = 61
    READ_TIMEOUT = 62
    TIMEOUT = 63
    DNS_LOOKUP_FAILED = 64
    TLS_PROTOCOL_ERROR = 65
    URL_PARSING_ERROR = 66
    NETWORK_IO_ERROR = 67
    FILE_IO_ERROR = 68
    INVALID_JSON_RESPONSE = 69
    NO_DATA = 70
    API_GATEWAY_ERROR = 71

    def get_error_message(self, error_code):
        """
        Get description of error code

        Parameters
        ----------
        error_code : int
            APIError error code

        Returns
        -------
        string
        """

        error_codes = {
            self.INVALID_PARAMETER: "MISSING OR INVALID PARAMETER",
            self.MAX_CALL_LIMIT: "DAILY API LIMIT EXCEEDED",
            self.BAD_URL: "INVALID URL",
            self.ABUSE_DETECTED: "ACCOUNT OR IP BANNED",
            self.NOT_RESPONDING: "NOT RESPONDING. RETRY IN 5 SECONDS",
            self.CONCURRENT: "TOO MANY CONNECTIONS",
            self.NOT_VERIFIED: "ACCOUNT NOT VERIFIED",
            self.TELEPHONY_LIMIT: "TELEPHONY NOT ENABLED ON YOUR ACCOUNT. PLEASE CONTACT SUPPORT FOR HELP",
            self.INVALID_JSON: "INVALID JSON. JSON CONTENT TYPE SET BUT NON-PARSABLE JSON SUPPLIED",
            self.ACCESS_DENIED: "ACCESS DENIED. PLEASE CONTACT SUPPORT FOR ACCESS TO THIS API",
            self.MAX_PHONE_CALLS: "MAXIMUM SIMULTANEOUS PHONE CALLS",
            self.BAD_AUDIO: "COULD NOT LOAD AUDIO FROM URL",
            self.HLR_LIMIT_REACHED: "HLR LIMIT REACHED. CARD DECLINED",
            self.TELEPHONY_BLOCKED: "CALLS AND SMS TO THIS NUMBER ARE LIMITED",
            self.TELEPHONY_RATE_EXCEEDED: "CALL IN PROGRESS",
            self.FREE_LIMIT: "FREE PLAN LIMIT EXCEEDED",
            self.RENDERING_FAILED: "RENDERING FAILED. COULD NOT GENERATE OUTPUT FILE",
            self.DEPRECATED_API: "THIS API IS DEPRECATED. PLEASE USE THE LATEST VERSION",
            self.CREDIT_LIMIT_REACHED: "MAXIMUM ACCOUNT CREDIT LIMIT REACHED. PAYMENT METHOD DECLINED",
            self.NOT_MULTI_ENABLED: "BATCH PROCESSING NOT ENABLED FOR THIS ENDPOINT",
            self.NO_BATCH_MODE: "BATCH PROCESSING NOT AVAILABLE ON YOUR PLAN",
            self.BATCH_LIMIT_EXCEEDED: "BATCH PROCESSING REQUEST LIMIT EXCEEDED",
            self.BATCH_INVALID: "INVALID BATCH REQUEST. DOES NOT CONFORM TO SPEC",
            self.DATASET_UNKNOWN: "UNKNOWN DATASET ID OR INVALID DATASET OPTION SUPPLIED",
            self.DATASET_GENERATION_FAILED: "FAILED TO GENERATE DATASET OUTPUT FILE",
            self.DATASET_INVALID_DATE: "INVALID DATE SUPPLIED OR DATE IS BEFORE FIRST HISTORICAL RECORD",
            self.USER_DEFINED_DAILY_LIMIT: "DAILY API LIMIT EXCEEDED. SET BY ACCOUNT HOLDER",
            self.ACCESS_FORBIDDEN: "ACCESS DENIED. USER ID OR API KEY INVALID",
            self.REQUEST_TOO_LARGE: "REQUEST TOO LARGE. MAXIMUM SIZE IS 5MB FOR DATA AND 25MB FOR UPLOADS",
            self.NO_ENDPOINT: "ENDPOINT DOES NOT EXIST",
            self.INTERNAL_SERVER_ERROR: "FATAL EXCEPTION. REQUEST COULD NOT BE COMPLETED",
            self.SERVER_OFFLINE: "SERVER OFFLINE. MAINTENANCE IN PROGRESS",
            self.CONNECT_TIMEOUT: "TIMEOUT OCCURRED CONNECTING TO SERVER",
            self.READ_TIMEOUT: "TIMEOUT OCCURRED READING API RESPONSE",
            self.TIMEOUT: "TIMEOUT OCCURRED DURING API REQUEST",
            self.DNS_LOOKUP_FAILED: "ERROR RECEIVED FROM YOUR DNS RESOLVER",
            self.TLS_PROTOCOL_ERROR: "ERROR DURING TLS PROTOCOL HANDSHAKE",
            self.URL_PARSING_ERROR: "ERROR PARSING REQUEST URL",
            self.NETWORK_IO_ERROR: "IO ERROR DURING API REQUEST",
            self.FILE_IO_ERROR: "IO ERROR WRITING TO OUTPUT FILE",
            self.INVALID_JSON_RESPONSE: "INVALID JSON DATA RECEIVED",
            self.NO_DATA: "NO PAYLOAD DATA RECEIVED",
            self.API_GATEWAY_ERROR: "API GATEWAY ERROR"
        }

        return error_codes.get(error_code, "API Error: {0}".format(error_code))
